﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

namespace Inet.Viewer.Data
{
    /// <summary>
    /// Progress for loading a report from the server. Pings the server
    /// each 5 seconds to obtain the current rendering state.
    /// </summary>
    public class LoadProgress : Progress
    {
        private const int InitialDelay = 1000;
        private const int Interval = 5000;
        private IRenderData ReportData;
        private Object sleepMonitor = new Object();

        /// <summary>
        /// Create a new progress.
        /// </summary>
        /// <param name="errorDelegate">for passing in an error (not used here)</param>
        /// <param name="ReportData">the data instance of the report</param>
        public LoadProgress(Action<Exception> errorDelegate, IRenderData ReportData) :
            base(errorDelegate, ProgressType.PageLoader)
        {
            this.ReportData = ReportData;
            this.ReportState = new ReportState(Inet.Viewer.Resources.strings.ProgressLoading, 0);
            this.ProgressMode = Data.ProgressMode.Steps;
            TotalProgress = 100;
        }

        /// <summary>
        /// Gets the render state instance.
        /// </summary>
        public ReportState ReportState { get; private set; }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override string Name
        {
            get
            {
                return ReportState.Text;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        protected override void Run()
        {
            lock (sleepMonitor)
            {
                Monitor.Wait(sleepMonitor, InitialDelay);
            }
            while (Status == ProgressStatus.Running)
            {
                try
                {
                    ReportState renderState = ReportData.Ping();

                    if (renderState != null)
                    {
                        ReportState = renderState;
                        ProgressCount = renderState.Progress;
                        OnProgressChanged();
                        if (renderState.Progress == 100)
                        {
                            Status = ProgressStatus.Completed;
                            return;
                        }
                    }
                    if (Status != ProgressStatus.Running)
                    {
                        break;
                    }
                    lock (sleepMonitor)
                    {
                        Monitor.Wait(sleepMonitor, Interval);
                    }
                }
                catch (Exception e)                
                {
                    ShowError(e);
                    Status = ProgressStatus.Error;
                }
            } 
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override void Cancel()
        {
            if (Status != ProgressStatus.Canceled)
            {
                Status = ProgressStatus.Canceled;
                ReportData.Stop();
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override ProgressStatus Status
        {
            set
            {
                base.Status = value;
                lock (sleepMonitor)
                {
                    Monitor.PulseAll(sleepMonitor);
                }
            }
        }
    }
}
